﻿# --[ Export SSRS RDL Report Queries   ]
# --[ Matt Balzan - Microsoft DSE 2023 ]
# --[ https://www.mattgpt.co.uk        ]

<# --[ 

    Description: Exports each and every query listed in each .rdl file from the SSRS Reports directory.
                 All exported files are saved as .sql files.
    
    Output:      Each query is labeled with a number like this example:
                 
                 Query #3
                 select dbo.fn_rbac_GetAdminIDsfromUserSIDs(@UserTokenSIDs) as UserSIDs

    How to use:  Edit the Source & Output directories and then save this script.
                 Run the script on the same device where the SSRS rdl files are located.
                 Default output will save to logged on user session desktop.
                    
#>


# --[ Specify the path to your RDL files ]
$sourceDirectory = "S:\Program Files\SMS_SRSRP\Reports\"

# --[ Specify the output directory ]
$outputDirectory = "~\Desktop\Exported-RDL-Reports"


# --[ DO NOT EDIT BELOW! ]

# --[ Initialize an empty string to store all CommandText sections ]
$allCommandText = ""
$rdlFiles = ""
$q = 1

# --[ Recursively find all RDL files in the source directory ]
$rdlFiles = Get-ChildItem -Path $sourceDirectory -File -Filter *.rdl -Recurse

# --[ Loop through each RDL file ]
foreach ($rdlFile in $rdlFiles) {

    # --[ Create the output path for the query XML section ]
    $relativePath = $rdlFile.FullName.Substring($sourceDirectory.Length)
    $outputPath = Join-Path -Path $outputDirectory -ChildPath ($relativePath -replace '\.rdl$', '.sql')

    # --[ Ensure the output folder exists, create it if not ]
    $outputFolder = Split-Path -Path $outputPath
    if (-not (Test-Path -Path $outputFolder -PathType Container)) {
        New-Item -Path $outputFolder -ItemType Directory -Force
    }

    # --[ Read the RDL file content ]
    $rdlContent = Get-Content -Path $rdlFile.FullName

    # --[ Regex to extract all <CommandText> sections ]
    $commandTextSections = [regex]::Matches($rdlContent, "<CommandText>(.*?)</CommandText>")

    # --[ Loop through each <CommandText> section found ]
    foreach ($match in $commandTextSections) {

        $allCommandText += "--Query #$q`r`n"
        $commandTextSection = $match.Groups[1].Value
        $allCommandText += $commandTextSection + "`r`n`r`n"
        $q++

           
    }

# --[ Save all CommandText sections to a single .sql file ]
$allCommandText | Set-Content -Path $outputPath 
Write-Output "All <CommandText> sections extracted and saved to $outputPath"

# --[ Reset CommandText & counter vars ]
$allCommandText = ""
$q = 1
}

# --[ End of script ]